////////////////////////////////////////////////////////////////////////////////
// DBTREEVIEW98                                                               //
////////////////////////////////////////////////////////////////////////////////
// Virtual DB Tree view for D3                                                //
// * Icons, Hottracking, new CustomDraw implementation, ... and more          //
////////////////////////////////////////////////////////////////////////////////
// Version 0.5 Beta                                                           //
// Date de cration           : 24/06/1997                                    //
// Date dernire modification : 26/06/1997                                    //
////////////////////////////////////////////////////////////////////////////////
// Jean-Luc Mattei                                                            //
// jlucm@club-internet.fr                                                     //
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// IMPORTANT NOTICE :                                                         //
//                                                                            //
//                                                                            //
// This program is FreeWare                                                   //
//                                                                            //
// Please do not release modified versions of this source code.               //
// If you've made any changes that you think should have been there,          //
// feel free to submit them to me at jlucm@club-internet.fr                   //
////////////////////////////////////////////////////////////////////////////////
// NOTES :                                                                    //
//                                                                            //
// * CheckBoxes will be implemented in the future                             //
// * flicks too much when scrolled                                            //
////////////////////////////////////////////////////////////////////////////////
//  REVISIONS :                                                               //
//                                                                            //
//  V 0.5 : * Added field type validation                                     //
//          * Added GetColor (utilcolr.pas) color index compatible with       //
//            Borland's TColorGrid                                            //
//  V 0.6 :                                                                   //
//        * All error messages are constants now                              //
//          Two langages supported french (define FRANCAIS) & english         //
//          (define ENGLISH) (not very hard to add more and to                //
//           correct english)                                                 //
//          thanks to Glen Verran he reminds me that french isn't universal :)//
//        * Lookup removed and replaced by Locate                             //
//        * Icon Field Name is no more absolutely needed                      //
//  V 0.7 :                                                                   //
//        * Property Tooltips published.                                      //
////////////////////////////////////////////////////////////////////////////////

{$DEFINE FRANCAIS}
//{$DEFINE ENGLISH}

unit DBTreeView98;

{$R *.DCR}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, CommCtrl, Db, DbCtrls, DBTables;

Type

  TCustomDBTreeView = class;

  TTreeViewDataLink = class(TDataLink)
  private
    FTreeView: TCustomDBTreeView;
    FFieldCount: Integer;
    FFieldMapSize: Integer;
    FFieldMap: Pointer;
    FModified: Boolean;
    FInUpdateData: Boolean;
    FSparseMap: Boolean;
    function GetDefaultFields: Boolean;
    function GetFields(I: Integer): TField;
  protected
    procedure ActiveChanged; override;
    procedure DataSetChanged; override;
    procedure DataSetScrolled(Distance: Integer); override;
    procedure FocusControl(Field: TFieldRef); override;
    procedure EditingChanged; override;
    procedure RecordChanged(Field: TField); override;
    procedure UpdateData; override;
    function  GetMappedIndex(ColIndex: Integer): Integer;
  public
    constructor Create(ATreeView: TCustomDBTreeView);
    destructor Destroy; override;
    function AddMapping(const FieldName: string): Boolean;
    procedure ClearMapping;
    procedure Modified;
    procedure Reset;
    property DefaultFields: Boolean read GetDefaultFields;
    property FieldCount: Integer read FFieldCount;
    property Fields[I: Integer]: TField read GetFields;
    property SparseMap: Boolean read FSparseMap write FSparseMap;
  end;

{ TCustomDBTreeView }

  ETreeViewError = class(Exception);

  TDBTVChangingEvent = procedure(Sender: TObject; Node: HTreeItem;
    var AllowChange: Boolean) of object;
  TDBTVChangedEvent = procedure(Sender: TObject; Node: HTreeItem) of object;
  TDBTVEditingEvent = procedure(Sender: TObject; Node: HTreeItem;
    var AllowEdit: Boolean) of object;
  TDBTVEditedEvent = procedure(Sender: TObject; Node: HTreeItem; var S: string) of object;
  TDBTVExpandingEvent = procedure(Sender: TObject; Node: HTreeItem;
    var AllowExpansion: Boolean) of object;
  TDBTVCollapsingEvent = procedure(Sender: TObject; Node: HTreeItem;
    var AllowCollapse: Boolean) of object;
  TDBTVExpandedEvent = procedure(Sender: TObject; Node: HTreeItem) of object;
  TDBTVCompareEvent = procedure(Sender: TObject; Node1, Node2: HTreeItem;
    Data: Integer; var Compare: Integer) of object;

  TSortType = (stNone, stData, stText, stBoth);

  TCustomDBTreeView = class(TWinControl)
  private
    FDetailTable: TTable;
    FDataLink: TTreeViewDataLink;
    FMasterFieldName: string;
    FDetailFieldName: string;
    FColorFieldName: string;
    FItemFieldName: string;
    FIconFieldName: string;
    FStartMasterValue: string;
    FKeepColor: Integer;
    FKeepImage: Integer;
    FKeepText: String;
    FKeepChildCount: Integer;
    FShowLines: Boolean;
    FShowRoot: Boolean;
    FShowButtons: Boolean;
    FBorderStyle: TBorderStyle;
    FReadOnly: Boolean;
    FImages: TImageList;
    FStateImages: TImageList;
    FImageChangeLink: TChangeLink;
    FStateChangeLink: TChangeLink;
    FDragImage: TImageList;
    FOldFrom: Integer;
    FOldSelection: Integer;
    FOldActive: Integer;
    FSortType: TSortType;
    FTooltips: Boolean;
    FSaveIndent: Integer;
    FHideSelection: Boolean;
    FEditInstance: Pointer;
    FCheckboxes: Boolean;
    FHotTrack: Boolean;
    FDefEditProc: Pointer;
    FEditHandle: HWND;
    FDragged: Boolean;
    FRClickNode: HTreeItem;
    FLastDropTarget: HTreeItem;
    FDragNode: HTreeItem;
    FManualNotify: Boolean;
    FRightClickSelect: Boolean;
    FStateChanging: Boolean;
    FWideText: WideString;
    FUpdateLock: Boolean;
    FOnEditing: TDBTVEditingEvent;
    FOnEdited: TDBTVEditedEvent;
    FOnExpanded: TDBTVExpandedEvent;
    FOnExpanding: TDBTVExpandingEvent;
    FOnCollapsed: TDBTVExpandedEvent;
    FOnCollapsing: TDBTVCollapsingEvent;
    FOnChanging: TDBTVChangingEvent;
    FOnChange: TDBTVChangedEvent;
    FOnCompare: TDBTVCompareEvent;
    FOnDeletion: TDBTVExpandedEvent;
    //FOnGetImageIndex: TDBTVExpandedEvent;
    //FOnGetSelectedIndex: TDBTVExpandedEvent;
    procedure ShowEditor;
    procedure EditingChanged;
    procedure HideEditor;
    function  AcquireFocus: Boolean;
    procedure UpdateActive;
    //procedure UpdateRowCount;
    procedure DataChanged;
    function  GetDataSource: TDataSource;
    procedure UpdateData;
    procedure LinkActive(Value: Boolean);
    procedure ScrollData(Distance: Integer);
    function  SelectedField: TField;
    procedure SetDataSource(Value: TDataSource);
    procedure CMColorChanged(var Message: TMessage); message CM_COLORCHANGED;
    procedure CMCtl3DChanged(var Message: TMessage); message CM_CTL3DCHANGED;
    procedure CMDrag(var Message: TCMDrag); message CM_DRAG;
    procedure CNNotify(var Message: TWMNotify); message CN_NOTIFY;
    procedure EditWndProc(var Message: TMessage);
    procedure DoDragOver(Source: TDragObject; X, Y: Integer; CanDrop: Boolean);
    //procedure GetImageIndex(Node: HTreeItem);
    //procedure GetSelectedIndex(Node: HTreeItem);
    function  GetDropTarget: HTreeItem;
    function  GetIndent: Integer;
    function  GetSelection: HTreeItem;
    function  GetTopItem: HTreeItem;
    procedure ImageListChange(Sender: TObject);
    procedure SetBorderStyle(Value: TBorderStyle);
    procedure SetButtonStyle(Value: Boolean);
    procedure SetCheckboxes(Value: Boolean);
    procedure SetDropTarget(Value: HTreeItem);
    procedure SetHideSelection(Value: Boolean);
    procedure SetHotTrack(Value: Boolean);
    procedure SetImageList(Value: HImageList; Flags: Integer);
    procedure SetIndent(Value: Integer);
    procedure SetImages(Value: TImageList);
    procedure SetLineStyle(Value: Boolean);
    procedure SetReadOnly(Value: Boolean);
    procedure SetRootStyle(Value: Boolean);
    procedure SetSelection(Value: HTreeItem);
    procedure SetSortType(Value: TSortType);
    procedure SetStateImages(Value: TImageList);
    procedure SetStyle(Value: Integer; UseStyle: Boolean);
    procedure SetToolTips(Value: Boolean);
    procedure SetTopItem(Value: HTreeItem);
    procedure WMLButtonDown(var Message: TWMLButtonDown); message WM_LBUTTONDOWN;
    procedure WMRButtonDown(var Message: TWMRButtonDown); message WM_RBUTTONDOWN;
    procedure WMRButtonUp(var Message: TWMRButtonUp); message WM_RBUTTONUP;
    procedure WMNotify(var Message: TWMNotify); message WM_NOTIFY;
    procedure CMSysColorChange(var Message: TMessage); message CM_SYSCOLORCHANGE;
    procedure ExpandItem(Item: HTreeItem; Recurse: Boolean);
    procedure CollapseItem(Item: HTreeItem; Recurse: Boolean);
    //function  GetNextItem(Item: HTreeItem): HTreeItem;
    procedure DeleteItem(Item: HTreeItem);
    function  HasAsParent(Item, Value: HTreeItem): Boolean;
    function  GetChildren(Item: HTreeItem): Boolean;
    function  GetItemState(Item: HTreeItem; NodeState: TNodeState): Boolean;
    procedure FocusItem(Item: HTreeItem; Value: Boolean);
    //function  GetChildrenCount(Item: HTreeItem): Integer;
    function  GetItem(Item: HTreeItem): TTVItem;
    procedure RecordChanged(Field: TField);
    function   ValidDataSet: Boolean;
    procedure SetMasterFieldName(Value: String);
    procedure SetColorFieldName(Value: String);
    procedure SetDetailFieldName(Value: String);
    procedure SetItemFieldName(Value: String);
    procedure SetIconFieldName(Value: String);
  protected
    FAcquireFocus: Boolean;
    function CanEdit(Node: HTreeItem): Boolean; dynamic;
    function CanChange(Node: HTreeItem): Boolean; dynamic;
    function CanCollapse(Node: HTreeItem): Boolean; dynamic;
    function CanExpand(Node: HTreeItem): Boolean; dynamic;
    procedure Change(Node: HTreeItem); dynamic;
    procedure Collapse(Node: HTreeItem); dynamic;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
    procedure DestroyWnd; override;
    procedure DoEndDrag(Target: TObject; X, Y: Integer); override;
    procedure DoStartDrag(var DragObject: TDragObject); override;
    procedure Edit(const Item: TTVItem); dynamic;
    procedure Expand(Node: HTreeItem); dynamic;
    function  GetDragImages: TCustomImageList; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure SetDragMode(Value: TDragMode); override;
    procedure WndProc(var Message: TMessage); override;
    property OnEditing: TDBTVEditingEvent read FOnEditing write FOnEditing;
    property OnEdited: TDBTVEditedEvent read FOnEdited write FOnEdited;
    property OnExpanding: TDBTVExpandingEvent read FOnExpanding write FOnExpanding;
    property OnExpanded: TDBTVExpandedEvent read FOnExpanded write FOnExpanded;
    property OnCollapsing: TDBTVCollapsingEvent read FOnCollapsing write FOnCollapsing;
    property OnCollapsed: TDBTVExpandedEvent read FOnCollapsed write FOnCollapsed;
    property OnChanging: TDBTVChangingEvent read FOnChanging write FOnChanging;
    property OnChange: TDBTVChangedEvent read FOnChange write FOnChange;
    property OnCompare: TDBTVCompareEvent read FOnCompare write FOnCompare;
    property OnDeletion: TDBTVExpandedEvent read FOnDeletion write FOnDeletion;
    //property OnGetImageIndex: TDBTVExpandedEvent read FOnGetImageIndex write FOnGetImageIndex;
    //property OnGetSelectedIndex: TDBTVExpandedEvent read FOnGetSelectedIndex write FOnGetSelectedIndex;
    property ShowButtons: Boolean read FShowButtons write SetButtonStyle default True;
    property BorderStyle: TBorderStyle read FBorderStyle write SetBorderStyle default bsSingle;
    property ShowLines: Boolean read FShowLines write SetLineStyle default True;
    property ShowRoot: Boolean read FShowRoot write SetRootStyle default True;
    property ReadOnly: Boolean read FReadOnly write SetReadOnly default False;
    property RightClickSelect: Boolean read FRightClickSelect write FRightClickSelect default False;
    property Indent: Integer read GetIndent write SetIndent;
    property SortType: TSortType read FSortType write SetSortType default stNone;
    property HideSelection: Boolean read FHideSelection write SetHideSelection default True;
    property Images: TImageList read FImages write SetImages;
    property StateImages: TImageList read FStateImages write SetStateImages;
  public
    constructor Create(AOwner: TComponent); override;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    destructor Destroy; override;
    function AddItem(Parent, Target: HTreeItem; Code: Longint): HTreeItem;
    function AlphaSort: Boolean;
    function CustomSort(SortProc: TTVCompare; Data: Longint): Boolean;
    function FindItemData(Code: Longint): HTreeItem;
    procedure FullCollapse;
    procedure FullExpand;
    property MasterFieldName: string read FMasterFieldName write SetMasterFieldName;
    property ColorFieldName: string read FColorFieldName write SetColorFieldName;
    property DetailFieldName: string read FDetailFieldName write SetDetailFieldName;
    property ItemFieldName: string read FItemFieldName write SetItemFieldName;
    property IconFieldName: string read FIconFieldName write SetIconFieldName;
    property StartMasterValue: string read FStartMasterValue write FStartMasterValue;
    function GetHitTestInfoAt(X, Y: Integer): THitTests;
    function GetNodeAt(X, Y: Integer): HTreeItem;
    property DataLink: TTreeViewDataLink read FDataLink;
    property HotTrack: Boolean read FHotTrack write SetHotTrack default False;
    function IsEditing: Boolean;
    property Checkboxes: Boolean read FCheckboxes write SetCheckboxes default False;
    property DropTarget: HTreeItem read GetDropTarget write SetDropTarget;
    property Selected: HTreeItem read GetSelection write SetSelection;
    property TopItem: HTreeItem read GetTopItem write SetTopItem;
    property ToolTips: Boolean read FToolTips write SetToolTips;
  end;

  TDBTreeView = class(TCustomDBTreeView)
  published
    property DataSource;
    property ShowButtons;
    property BorderStyle;
    property DragCursor;
    property ShowLines;
    property ShowRoot;
    property ReadOnly;
    property RightClickSelect;
    property DragMode;
    property HideSelection;
    property Indent;
    property MasterFieldName;
    property ColorFieldName;
    property DetailFieldName;
    property IconFieldName;
    property ItemFieldName;
    property StartMasterValue;
    property OnEditing;
    property OnEdited;
    property OnExpanding;
    property OnExpanded;
    property OnCollapsing;
    property OnCompare;
    property OnCollapsed;
    property OnChanging;
    property OnChange;
    property OnDeletion;
    //property OnGetImageIndex;
    //property OnGetSelectedIndex;
    property Align;
    property Enabled;
    property Font;
    property Checkboxes;
    property Color;
    property HotTrack;
    property ParentColor default False;
    property ParentCtl3D;
    property Ctl3D;
    property SortType;
    property TabOrder;
    property TabStop default True;
    property ToolTips;
    property Visible;
    property OnClick;
    property OnEnter;
    property OnExit;
    property OnDragDrop;
    property OnDragOver;
    property OnStartDrag;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnDblClick;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property PopupMenu;
    property ParentFont;
    property ParentShowHint;
    property ShowHint;
    property Images;
    property StateImages;
  end;

procedure Register;

implementation

Uses ComCtl98, DBConsts, DBGrids, DsgnIntf, TypInfo, UtilColr;

const
{$ifdef FRANCAIS}
  StrMasterFieldNameEmpty  = 'La proprit "MasterFieldName" doit tre renseigne';
  StrDetailFieldNameEmpty  = 'La proprit "DetailFieldName" doit tre renseigne';
  StrItemFieldNameEmpty    = 'La proprit "ItemFieldName" doit tre renseigne';
  //StrIconFieldNameEmpty  = 'La proprit "IconFieldName" doit tre renseigne';
  StrMasterDetailFieldNameError = 'Les champs "MasterFieldName" et "DetailFieldName" doivent tre du mme type';
  StrMasterFieldNameError  = 'Le champ "MasterFieldName" doit tre de type entier';
  StrDetailFieldNameError  = 'Le champ "DetailFieldName" doit tre de type entier';
  StrItemFieldNameError    = 'Le champ "ItemFieldName" doit tre de type chaine, date ou entier';
  StrIconFieldNameError    = 'Le champ "IconFieldName" doit tre de type entier';
  StrColorFieldNameError   = 'Le champ "ColorFieldName" doit tre de type entier';
{$endif}
{$ifdef ENGLISH}
  StrMasterFieldNameEmpty  = '"MasterFieldName" property must be filled';
  StrDetailFieldNameEmpty  = '"DetailFieldName" property must be filled';
  StrItemFieldNameEmpty    = '"ItemFieldName" property must be filled';
  //StrIconFieldNameEmpty  = '"IconFieldName" property must be filled';
  StrMasterDetailFieldNameError = '"MasterFieldName" and "DetailFieldName" must be of same type';
  StrMasterFieldNameError  = '"MasterFieldName" must be integer type';
  StrDetailFieldNameError  = '"DetailFieldName" must be integer type';
  StrItemFieldNameError    = '"ItemFieldName" must be string, date or integer type';
  StrIconFieldNameError    = '"IconFieldName" must be integer type';
  StrColorFieldNameError   = '"ColorFieldName" must be integer type';
{$endif}

type

  TDBStringProperty = class(TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValueList(List: TStrings); virtual; abstract;
    procedure GetValues(Proc: TGetStrProc); override;
  end;

function TDBStringProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList, paSortList, paMultiSelect];
end;

procedure TDBStringProperty.GetValues(Proc: TGetStrProc);
var
  I: Integer;
  Values: TStringList;
begin
  Values := TStringList.Create;
  try
    GetValueList(Values);
    for I := 0 to Values.Count - 1 do Proc(Values[I]);
  finally
    Values.Free;
  end;
end;

type

  TDataFieldProperty = class(TDBStringProperty)
  public
    function GetDataSourcePropName: string; virtual;
    procedure GetValueList(List: TStrings); override;
  end;

function TDataFieldProperty.GetDataSourcePropName: string;
begin
  Result := 'DataSource';
end;

procedure TDataFieldProperty.GetValueList(List: TStrings);
var
  Instance: TComponent;
  PropInfo: PPropInfo;
  DataSource: TDataSource;
begin
  Instance := TComponent(GetComponent(0));
  PropInfo := TypInfo.GetPropInfo(Instance.ClassInfo, GetDataSourcePropName);
  if (PropInfo <> nil) and (PropInfo^.PropType^.Kind = tkClass) then
  begin
    DataSource := TObject(GetOrdProp(Instance, PropInfo)) as TDataSource;
    if (DataSource <> nil) and (DataSource.DataSet <> nil) then
      DataSource.DataSet.GetFieldNames(List);
  end;
end;

procedure Register;
begin
  RegisterComponents('Exemples', [TDBTreeView]);
  RegisterPropertyEditor(TypeInfo(string), TDBTreeView, 'ColorFieldName', TDataFieldProperty);
  RegisterPropertyEditor(TypeInfo(string), TDBTreeView, 'IconFieldName', TDataFieldProperty);
  RegisterPropertyEditor(TypeInfo(string), TDBTreeView, 'ItemFieldName', TDataFieldProperty);
  RegisterPropertyEditor(TypeInfo(string), TDBTreeView, 'MasterFieldName', TDataFieldProperty);
  RegisterPropertyEditor(TypeInfo(string), TDBTreeView, 'DetailFieldName', TDataFieldProperty);
end;

const
  MaxMapSize = (MaxInt div 2) div SizeOf(Integer);  { 250 million }

type
  TIntArray = array[0..MaxMapSize] of Integer;
  PIntArray = ^TIntArray;

procedure KillMessage(Wnd: HWnd; Msg: Integer);
// Delete the requested message from the queue, but throw back
// any WM_QUIT msgs that PeekMessage may also return
var
  M: TMsg;
begin
  M.Message := 0;
  if PeekMessage(M, Wnd, Msg, Msg, pm_Remove) and (M.Message = WM_QUIT) then
    PostQuitMessage(M.wparam);
end;

{ Error reporting }

procedure RaiseTreeViewError(const S: string);
begin
  raise EInvalidOperation.Create(S);
end;

{ TTreeViewDataLink }

constructor TTreeViewDataLink.Create(ATreeView: TCustomDBTreeView);
begin
  inherited Create;
  FTreeView := ATreeView;
end;

destructor TTreeViewDataLink.Destroy;
begin
  ClearMapping;
  inherited Destroy;
end;

function TTreeViewDataLink.GetDefaultFields: Boolean;
var
  I: Integer;
begin
  Result := True;
  if DataSet <> nil then Result := DataSet.DefaultFields;
  if Result and SparseMap then
  for I := 0 to FFieldCount-1 do
    if PIntArray(FFieldMap)^[I] < 0 then
    begin
      Result := False;
      Exit;
    end;
end;

function TTreeViewDataLink.GetFields(I: Integer): TField;
begin
  if (0 <= I) and (I < FFieldCount) and (PIntArray(FFieldMap)^[I] >= 0) then
    Result := DataSet.Fields[PIntArray(FFieldMap)^[I]]
  else
    Result := nil;
end;

function TTreeViewDataLink.AddMapping(const FieldName: string): Boolean;
var
  Field: TField;
  NewSize: Integer;
begin
  Result := True;
  if FFieldCount >= MaxMapSize then RaiseTreeViewError(STooManyColumns);
  if SparseMap then
    Field := DataSet.FindField(FieldName)
  else
    Field := DataSet.FieldByName(FieldName);

  if FFieldCount = FFieldMapSize then
  begin
    NewSize := FFieldMapSize;
    if NewSize = 0 then
      NewSize := 8
    else
      Inc(NewSize, NewSize);
    if (NewSize < FFieldCount) then
      NewSize := FFieldCount + 1;
    if (NewSize > MaxMapSize) then
      NewSize := MaxMapSize;
    ReallocMem(FFieldMap, NewSize * SizeOf(Integer));
    FFieldMapSize := NewSize;
  end;
  if Assigned(Field) then
  begin
    PIntArray(FFieldMap)^[FFieldCount] := Field.Index;
    Field.FreeNotification(FTreeView);
  end
  else
    PIntArray(FFieldMap)^[FFieldCount] := -1;
  Inc(FFieldCount);
end;

procedure TTreeViewDataLink.ActiveChanged;
begin
  FTreeView.LinkActive(Active);
end;

procedure TTreeViewDataLink.ClearMapping;
begin
  if FFieldMap <> nil then
  begin
    FreeMem(FFieldMap, FFieldMapSize * SizeOf(Integer));
    FFieldMap := nil;
    FFieldMapSize := 0;
    FFieldCount := 0;
  end;
end;

procedure TTreeViewDataLink.Modified;
begin
  FModified := True;
end;

procedure TTreeViewDataLink.DataSetChanged;
begin
  FTreeView.DataChanged;
  FModified := False;
end;

procedure TTreeViewDataLink.DataSetScrolled(Distance: Integer);
begin
  FTreeView.ScrollData(Distance);
end;

procedure TTreeViewDataLink.FocusControl(Field: TFieldRef);
begin
  if Assigned(Field) and Assigned(Field^) then
  begin
    if (FTreeView.SelectedField = Field^) and FTreeView.AcquireFocus then
    begin
      Field^ := nil;
      FTreeView.ShowEditor;
    end;
  end;
end;

procedure TTreeViewDataLink.EditingChanged;
begin
  FTreeView.EditingChanged;
end;

procedure TTreeViewDataLink.RecordChanged(Field: TField);
begin
  FTreeView.RecordChanged(Field);
  FModified := False;
end;

procedure TTreeViewDataLink.UpdateData;
begin
  FInUpdateData := True;
  try
    if FModified then FTreeView.UpdateData;
    FModified := False;
  finally
    FInUpdateData := False;
  end;
end;

function TTreeViewDataLink.GetMappedIndex(ColIndex: Integer): Integer;
begin
  if (0 <= ColIndex) and (ColIndex < FFieldCount) then
    Result := PIntArray(FFieldMap)^[ColIndex]
  else
    Result := -1;
end;

procedure TTreeViewDataLink.Reset;
begin
  if FModified then RecordChanged(nil) else Dataset.Cancel;
end;

{ TCustomDBTreeView }

constructor TCustomDBTreeView.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FUpdateLock:= False;
  FOldFrom:= -1;
  FOldSelection:= -1;
  FKeepColor:= 0;
  FKeepImage:= -1;
  FKeepChildCount:= 0;
  FKeepText:= '';
  ControlStyle := ControlStyle - [csCaptureMouse] + [csDisplayDragImage, csReflector];
  FAcquireFocus := True;
  FDataLink := TTreeViewDataLink.Create(Self);
  Width := 121;
  Height := 97;
  TabStop := True;
  ParentColor := False;
  FDetailTable:= TTable.Create(Self);
  FBorderStyle := bsSingle;
  FShowButtons := True;
  FShowRoot := True;
  FShowLines := True;
  FHideSelection := True;
  FDragImage := TImageList.CreateSize(32, 32);
  FSaveIndent := -1;
  FEditInstance := MakeObjectInstance(EditWndProc);
  FImageChangeLink := TChangeLink.Create;
  FImageChangeLink.OnChange := ImageListChange;
  FStateChangeLink := TChangeLink.Create;
  FStateChangeLink.OnChange := ImageListChange;
end;

destructor TCustomDBTreeView.Destroy;
begin
  if ( ValidDataSet ) then
    FDataLink.ActiveRecord:= 0;
  FDataLink.Free;
  FDetailTable.Active:= False;
  FDetailTable.Free;
  FDragImage.Free;
  FreeObjectInstance(FEditInstance);
  FImageChangeLink.Free;
  FStateChangeLink.Free;
  inherited Destroy;
end;

procedure TCustomDBTreeView.CreateParams(var Params: TCreateParams);
const
  BorderStyles: array[TBorderStyle] of Integer = (0, WS_BORDER);
  LineStyles: array[Boolean] of Integer = (0, TVS_HASLINES);
  RootStyles: array[Boolean] of Integer = (0, TVS_LINESATROOT);
  ButtonStyles: array[Boolean] of Integer = (0, TVS_HASBUTTONS);
  EditStyles: array[Boolean] of Integer = (TVS_EDITLABELS, 0);
  HideSelections: array[Boolean] of Integer = (TVS_SHOWSELALWAYS, 0);
  DragStyles: array[TDragMode] of Integer = (TVS_DISABLEDRAGDROP, 0);
begin
  InitCommonControl(ICC_TREEVIEW_CLASSES);
  inherited CreateParams(Params);
  CreateSubClass(Params, WC_TREEVIEW);
  with Params do
  begin
    Style := Style or LineStyles[FShowLines] or BorderStyles[FBorderStyle] or
      RootStyles[FShowRoot] or ButtonStyles[FShowButtons] or
      EditStyles[FReadOnly] or HideSelections[FHideSelection] or
      DragStyles[DragMode];
    if ( HotTrack ) then Style:= Style or TVS_HOTTRACK;
    if ( CheckBoxes ) then Style:= Style or TVS_CHECKBOXES;
    if ( Not ToolTips ) then Style:= Style or TVS_NOTOOLTIPS;
    if Ctl3D and NewStyleControls and (FBorderStyle = bsSingle) then
    begin
      Style := Style and not WS_BORDER;
      ExStyle := Params.ExStyle or WS_EX_CLIENTEDGE;
    end;
    WindowClass.style := WindowClass.style and not (CS_HREDRAW or CS_VREDRAW);
  end;
end;

procedure TCustomDBTreeView.CreateWnd;
begin
  FStateChanging := False;
  inherited CreateWnd;
  if FSaveIndent <> -1 then Indent := FSaveIndent;
  if (Images <> nil) and Images.HandleAllocated then
    SetImageList(Images.Handle, TVSIL_NORMAL);
  if (StateImages <> nil) and StateImages.HandleAllocated then
    SetImageList(StateImages.Handle, TVSIL_STATE);
end;

procedure TCustomDBTreeView.DestroyWnd;
begin
  TreeView_DeleteAllItems(Handle);
  FStateChanging := True;
  FSaveIndent := Indent;
  inherited DestroyWnd;
end;

procedure TCustomDBTreeView.EditWndProc(var Message: TMessage);
begin
  try
    with Message do
    begin
      case Msg of
        WM_KEYDOWN,
        WM_SYSKEYDOWN: if DoKeyDown(TWMKey(Message)) then Exit;
        WM_CHAR: if DoKeyPress(TWMKey(Message)) then Exit;
        WM_KEYUP,
        WM_SYSKEYUP: if DoKeyUp(TWMKey(Message)) then Exit;
        CN_KEYDOWN,
        CN_CHAR, CN_SYSKEYDOWN,
        CN_SYSCHAR:
          begin
            WndProc(Message);
            Exit;
          end;
      end;
      Result := CallWindowProc(FDefEditProc, FEditHandle, Msg, WParam, LParam);
    end;
  except
    Application.HandleException(Self);
  end;
end;

procedure TCustomDBTreeView.CMColorChanged(var Message: TMessage);
begin
  inherited;
  RecreateWnd;
end;

procedure TCustomDBTreeView.CMCtl3DChanged(var Message: TMessage);
begin
  inherited;
  if FBorderStyle = bsSingle then RecreateWnd;
end;

procedure TCustomDBTreeView.CMSysColorChange(var Message: TMessage);
begin
  inherited;
  if not (csLoading in ComponentState) then
  begin
    Message.Msg := WM_SYSCOLORCHANGE;
    DefaultHandler(Message);
  end;
end;

function TCustomDBTreeView.AlphaSort: Boolean;
begin
  {if HandleAllocated then
  begin
    Result := CustomSort(nil, 0);
    for I := 0 to Items.Count - 1 do
      with Items[I] do
        if HasChildren then AlphaSort;
  end
  else }Result := False;
end;

function TCustomDBTreeView.CustomSort(SortProc: TTVCompare; Data: Longint): Boolean;
//var
  //SortCB: TTVSortCB;
  //I: Integer;
  //Node: TTreeNode;
begin
  Result := False;
  {if HandleAllocated then
  begin
    with SortCB do
    begin
      if not Assigned(SortProc) then lpfnCompare := @DefaultTreeViewSort
      else lpfnCompare := SortProc;
      hParent := TVI_ROOT;
      lParam := Data;
      Result := TreeView_SortChildrenCB(Handle, SortCB, 0);
    end;
    for I := 0 to Items.Count - 1 do
    begin
      Node := Items[I];
      if Node.HasChildren then Node.CustomSort(SortProc, Data);
    end;
  end;}
end;

procedure TCustomDBTreeView.SetSortType(Value: TSortType);
begin
  if SortType <> Value then
  begin
    FSortType := Value;
    if ((SortType in [stData, stBoth]) and Assigned(OnCompare)) or
      (SortType in [stText, stBoth]) then
      AlphaSort;
  end;
end;

procedure TCustomDBTreeView.SetStyle(Value: Integer; UseStyle: Boolean);
var
  Style: Integer;
begin
  if HandleAllocated then
  begin
    Style := GetWindowLong(Handle, GWL_STYLE);
    if not UseStyle then Style := Style and not Value
    else Style := Style or Value;
    SetWindowLong(Handle, GWL_STYLE, Style);
  end;
end;

procedure TCustomDBTreeView.SetBorderStyle(Value: TBorderStyle);
begin
  if BorderStyle <> Value then
  begin
    FBorderStyle := Value;
    RecreateWnd;
  end;
end;

procedure TCustomDBTreeView.SetDragMode(Value: TDragMode);
begin
  if Value <> DragMode then
    SetStyle(TVS_DISABLEDRAGDROP, Value = dmManual);
  inherited;
end;

procedure TCustomDBTreeView.SetButtonStyle(Value: Boolean);
begin
  if ShowButtons <> Value then
  begin
    FShowButtons := Value;
    SetStyle(TVS_HASBUTTONS, Value);
  end;
end;

procedure TCustomDBTreeView.SetLineStyle(Value: Boolean);
begin
  if ShowLines <> Value then
  begin
    FShowLines := Value;
    SetStyle(TVS_HASLINES, Value);
  end;
end;

procedure TCustomDBTreeView.SetRootStyle(Value: Boolean);
begin
  if ShowRoot <> Value then
  begin
    FShowRoot := Value;
    SetStyle(TVS_LINESATROOT, Value);
  end;
end;

procedure TCustomDBTreeView.SetReadOnly(Value: Boolean);
begin
  if ReadOnly <> Value then
  begin
    FReadOnly := Value;
    SetStyle(TVS_EDITLABELS, not Value);
  end;
end;

procedure TCustomDBTreeView.SetHideSelection(Value: Boolean);
begin
  if HideSelection <> Value then
  begin
    FHideSelection := Value;
    SetStyle(TVS_SHOWSELALWAYS, not Value);
    Invalidate;
  end;
end;

function TCustomDBTreeView.GetNodeAt(X, Y: Integer): HTreeItem;
var
  HitTest: TTVHitTestInfo;
begin
  with HitTest do
  begin
    pt.X := X;
    pt.Y := Y;
    Result:= TreeView_HitTest(Handle, HitTest);
  end;
end;

function TCustomDBTreeView.GetHitTestInfoAt(X, Y: Integer): THitTests;
var
  HitTest: TTVHitTestInfo;
begin
  Result := [];
  with HitTest do
  begin
    pt.X := X;
    pt.Y := Y;
    TreeView_HitTest(Handle, HitTest);
    if (flags and TVHT_ABOVE) <> 0 then Include(Result, htAbove);
    if (flags and TVHT_BELOW) <> 0 then Include(Result, htBelow);
    if (flags and TVHT_NOWHERE) <> 0 then Include(Result, htNowhere);
    if (flags and TVHT_ONITEM) <> 0 then Include(Result, htOnItem);
    if (flags and TVHT_ONITEMBUTTON) <> 0 then Include(Result, htOnButton);
    if (flags and TVHT_ONITEMICON) <> 0 then Include(Result, htOnIcon);
    if (flags and TVHT_ONITEMINDENT) <> 0 then Include(Result, htOnIndent);
    if (flags and TVHT_ONITEMLABEL) <> 0 then Include(Result, htOnLabel);
    if (flags and TVHT_ONITEMRIGHT) <> 0 then Include(Result, htOnRight);
    if (flags and TVHT_ONITEMSTATEICON) <> 0 then Include(Result, htOnStateIcon);
    if (flags and TVHT_TOLEFT) <> 0 then Include(Result, htToLeft);
    if (flags and TVHT_TORIGHT) <> 0 then Include(Result, htToRight);
  end;
end;

procedure TCustomDBTreeView.SetIndent(Value: Integer);
begin
  if Value <> Indent then TreeView_SetIndent(Handle, Value);
end;

function TCustomDBTreeView.GetIndent: Integer;
begin
  Result := TreeView_GetIndent(Handle)
end;

procedure TCustomDBTreeView.FullExpand;
var
  Node: HTreeItem;
begin
  Node := TreeView_GetRoot(Handle);
  while Node <> nil do
  begin
    ExpandItem(Node, True);
    Node := TreeView_GetNextSibling(Handle, Node);
  end;
end;

procedure TCustomDBTreeView.FullCollapse;
var
  Node: HTreeItem;
begin
  Node := TreeView_GetRoot(Handle);
  while Node <> nil do
  begin
    CollapseItem(Node, True);
    Node := TreeView_GetNextSibling(Handle, Node);
  end;
end;

procedure TCustomDBTreeView.Loaded;
begin
  inherited Loaded;
end;

function TCustomDBTreeView.GetTopItem: HTreeItem;
begin
  if HandleAllocated then
    Result := TreeView_GetFirstVisible(Handle)
  else Result := nil;
end;

procedure TCustomDBTreeView.SetTopItem(Value: HTreeItem);
begin
  if HandleAllocated and (Value <> nil) then
    TreeView_SelectSetFirstVisible(Handle, Value);
end;

function TCustomDBTreeView.GetSelection: HTreeItem;
begin
  if HandleAllocated then
  begin
    if FRightClickSelect and Assigned(FRClickNode) then
      Result := FRClickNode
    else
      Result := TreeView_GetSelection(Handle);
  end
  else Result := nil;
end;

procedure TCustomDBTreeView.SetSelection(Value: HTreeItem);
begin
  if Value <> nil then TreeView_SelectItem(Handle, Value)
  else TreeView_SelectItem(Handle, nil);
end;

function TCustomDBTreeView.GetDropTarget: HTreeItem;
begin
  if HandleAllocated then
  begin
    Result := TreeView_GetDropHilite(Handle);
    if Result = nil then Result := FLastDropTarget;
  end
  else Result := nil;
end;

procedure TCustomDBTreeView.SetDropTarget(Value: HTreeItem);
begin
  if HandleAllocated then
    if Value <> nil then TreeView_SelectDropTarget(Handle, Value)
    else TreeView_SelectDropTarget(Handle, nil);
end;

function TCustomDBTreeView.IsEditing: Boolean;
var
  ControlHand: HWnd;
begin
  ControlHand := TreeView_GetEditControl(Handle);
  Result := (ControlHand <> 0) and IsWindowVisible(ControlHand);
end;

procedure TCustomDBTreeView.CNNotify(var Message: TWMNotify);
var
  Node: HTreeItem;
  MousePos: TPoint;
  //VarVal: Variant;
begin
  with Message.NMHdr^ do
    case code of
      NM_CUSTOMDRAW:
        begin
          with PNMCustomDrawInfo(Pointer(Message.NMHdr))^ do begin
            if ( dwDrawStage and CDDS_PREPAINT ) = CDDS_PREPAINT then
              Message.Result:= CDRF_NOTIFYITEMDRAW;
            if ( dwDrawStage and CDDS_ITEMPREPAINT ) = CDDS_ITEMPREPAINT then begin
              if ( uItemstate AND CDIS_SELECTED ) = 0 then
                SetTextColor(hdc, GetColor(FKeepColor))
              else begin
                FUpdateLock:= True;
                DataLink.DataSet.DisableControls;
                DataLink.DataSet.Locate(MasterFieldName, lItemlParam, [loCaseInsensitive, loPartialKey]);
                DataLink.DataSet.EnableControls;
                FUpdateLock:= False;
              end;
              Message.Result:= CDRF_NOTIFYITEMDRAW;
            end;
          end;
        end;
      TVN_BEGINDRAG:
        begin
          FDragged := True;
          with PNMTreeView(Pointer(Message.NMHdr))^ do
            FDragNode := ItemNew.hItem;
        end;
      TVN_BEGINLABELEDIT:
        begin
          with PTVDispInfo(Pointer(Message.NMHdr))^ do
            if Dragging or not CanEdit(item.hItem) then
              Message.Result := 1;
          if Message.Result = 0 then
          begin
            FEditHandle := TreeView_GetEditControl(Handle);
            FDefEditProc := Pointer(GetWindowLong(FEditHandle, GWL_WNDPROC));
            SetWindowLong(FEditHandle, GWL_WNDPROC, LongInt(FEditInstance));
          end;
        end;
      TVN_ENDLABELEDIT:
        with PTVDispInfo(Pointer(Message.NMHdr))^ do
          Edit(item);
      TVN_ITEMEXPANDING:
        if not FManualNotify then
        begin
          with PNMTreeView(Pointer(Message.NMHdr))^ do
          begin
            Node := ItemNew.hItem;
            if (action = TVE_EXPAND) and not CanExpand(Node) then
              Message.Result := 1
            else if (action = TVE_COLLAPSE) and
              not CanCollapse(Node) then Message.Result := 1;
          end;
        end;
      TVN_ITEMEXPANDED:
        if not FManualNotify then
        begin
          with PNMTreeView(Pointer(Message.NMHdr))^ do
          begin
            Node := itemNew.hItem;
            if (action = TVE_EXPAND) then Expand(Node)
            else if (action = TVE_COLLAPSE) then Collapse(Node);
          end;
        end;
      TVN_SELCHANGING:
        with PNMTreeView(Pointer(Message.NMHdr))^ do
          if not CanChange(itemNew.hItem) then
            Message.Result := 1;
      TVN_SELCHANGED:
        with PNMTreeView(Pointer(Message.NMHdr))^ do
          Change(itemNew.hItem);
      TVN_DELETEITEM:
        begin
          if not FStateChanging then
          begin
            with PNMTreeView(Pointer(Message.NMHdr))^ do
              Node := itemOld.hItem;
            if Node <> nil then
            begin
              //Delete(Node);
            end;
          end;
        end;
      TVN_SETDISPINFO:
        with PTVDispInfo(Pointer(Message.NMHdr))^ do
        begin
          //Node := item.hItem;
          //if (Node <> nil) and ((item.mask and TVIF_TEXT) <> 0) then
          //  Node.Text := item.pszText;
        end;
      TVN_GETDISPINFO:
        with PTVDispInfo(Pointer(Message.NMHdr))^ do
        begin
          Node := item.hItem;
          if ( Node <> nil ) and ( ValidDataSet ) then
          begin
            {if ( GetItem(Node).lParam <> FOldSelection ) and ( ( item.state AND TVIS_SELECTED ) = TVIS_SELECTED ) then begin
              FUpdateLock:= True;
              DataLink.DataSet.Locate(MasterFieldName, GetItem(Node).lParam, [loCaseInsensitive, loPartialKey]);
              FUpdateLock:= False;
              FOldSelection:= GetItem(Node).lParam;
            end;}
            if Not ( GetItem(Node).lParam = FOldFrom ) then begin
              try
                //BeginUpdate;
                FUpdateLock:= True;
                DataLink.DataSet.DisableControls;
                FOldActive:= DataLink.DataSet.FieldByName(MasterFieldName).AsInteger;
                if ( DataLink.DataSet.Locate(MasterFieldName, GetItem(Node).lParam, [loCaseInsensitive, loPartialKey]) ) then begin
                  FKeepText:= DataLink.DataSet.FieldByName(ItemFieldName).AsString;
                  if ( ColorFieldName <> '' ) then
                    FKeepColor:= DataLink.DataSet.FieldByName(ColorFieldName).AsInteger;
                  if ( IconFieldName <> '' ) then
                    FKeepImage:= DataLink.DataSet.FieldByName(IconFieldName).AsInteger;
                end
                else begin
                  FKeepText:= '?';
                  FKeepColor:= Font.Color;
                  FKeepImage:= -1;
                end;
                if ( GetItem(Node).lParam <> FOldSelection ) and ( ( item.state AND TVIS_SELECTED ) = TVIS_SELECTED ) then
                  FOldSelection:= GetItem(Node).lParam
                else
                  DataLink.DataSet.Locate(MasterFieldName, FOldActive, [loCaseInsensitive, loPartialKey]);
                DataLink.DataSet.EnableControls;
                //EndUpdate;
                FUpdateLock:= False;
                //VarVal:= DataLink.DataSet.Lookup(MasterFieldName, GetItem(Node).lParam, ItemFieldName+';'+IconFieldName+';'+ColorFieldName);
                //if ( Not VarIsNull(VarVal) ) then begin
                //  FKeepText:= VarToStr(VarVal[0]);
                //  if ( ColorFieldName <> '' ) then
                //    FKeepColor:= VarVal[2];
                //  FKeepImage:= VarVal[1];
                //end
                //else begin
                //  FKeepText:= '?';
                //  FKeepColor:= Font.Color;
                //  FKeepImage:= -1;
                //end;
              except
                FKeepText:= '?';
                FKeepColor:= Font.Color;
                FKeepImage:= -1;
              end;
            end;
            if (item.mask and TVIF_TEXT) <> 0 then
              StrLCopy(item.pszText, PChar(FKeepText), item.cchTextMax);
            if (item.mask and TVIF_IMAGE) <> 0 then
            begin
              item.iImage := FKeepImage;
            end;
            if (item.mask and TVIF_SELECTEDIMAGE) <> 0 then
            begin
              item.iSelectedImage := FKeepImage;
            end;
            if (item.mask and TVIF_CHILDREN) <> 0 then
            begin
              if Not ( GetItem(Node).lParam = FOldFrom ) then begin
                FDetailTable.Filtered:= False;
                if ( FDataLink.DataSet.Filter <> '' ) then
                  FDetailTable.Filter:= FDataLink.DataSet.Filter+' and '+FDetailFieldName+'='+IntToStr(item.lParam)
                else
                  FDetailTable.Filter:= FDetailFieldName+'='+IntToStr(item.lParam);
                FDetailTable.Filtered:= True;
                FKeepChildCount:= FDetailTable.RecordCount;
              end;
              item.cChildren := FKeepChildCount;
              //FOldFrom:= GetItem(Node).lParam;
            end;
            //FOldFrom:= GetItem(Node).lParam;
          end;
        end;
      NM_RCLICK:
        begin
          if RightClickSelect then
          begin
            GetCursorPos(MousePos);
            with PointToSmallPoint(ScreenToClient(MousePos)) do
            begin
              FRClickNode := GetNodeAt(X, Y);
              Perform(WM_RBUTTONUP, 0, MakeLong(X, Y));
            end;
          end
          else FRClickNode := Pointer(1);
        end;
    end;
end;

function TCustomDBTreeView.GetDragImages: TCustomImageList;
begin
  if FDragImage.Count > 0 then
    Result := FDragImage else
    Result := nil;
end;

procedure TCustomDBTreeView.WndProc(var Message: TMessage);
begin
  if not (csDesigning in ComponentState) and ((Message.Msg = WM_LBUTTONDOWN) or
    (Message.Msg = WM_LBUTTONDBLCLK)) and not Dragging and (DragMode = dmAutomatic) then
  begin
    if not IsControlMouseMsg(TWMMouse(Message)) then
    begin
      ControlState := ControlState + [csLButtonDown];
      Dispatch(Message);
    end;
  end
  else inherited WndProc(Message);
end;

procedure TCustomDBTreeView.DoStartDrag(var DragObject: TDragObject);
var
  ImageHandle: HImageList;
  DragNode: HTreeItem;
  P: TPoint;
begin
  inherited DoStartDrag(DragObject);
  DragNode := FDragNode;
  FLastDropTarget := nil;
  FDragNode := nil;
  if DragNode = nil then
  begin
    GetCursorPos(P);
    with ScreenToClient(P) do DragNode := GetNodeAt(X, Y);
  end;
  if DragNode <> nil then
  begin
    ImageHandle := TreeView_CreateDragImage(Handle, DragNode);
    if ImageHandle <> 0 then
      with FDragImage do
      begin
        Handle := ImageHandle;
        SetDragImage(0, 2, 2);
      end;
  end;
end;

procedure TCustomDBTreeView.DoEndDrag(Target: TObject; X, Y: Integer);
begin
  inherited DoEndDrag(Target, X, Y);
  FLastDropTarget := nil;
end;

procedure TCustomDBTreeView.CMDrag(var Message: TCMDrag);
begin
  inherited;
  with Message, DragRec^ do
    case DragMessage of
      dmDragMove: with ScreenToClient(Pos) do DoDragOver(Source, X, Y, Message.Result<>0);
      dmDragLeave:
        begin
          TDragObject(Source).HideDragImage;
          FLastDropTarget := DropTarget;
          DropTarget := nil;
          TDragObject(Source).ShowDragImage;
        end;
      dmDragDrop: FLastDropTarget := nil;
    end;
end;

procedure TCustomDBTreeView.DoDragOver(Source: TDragObject; X, Y: Integer; CanDrop: Boolean);
var
  Node: HTreeItem;
begin
  Node := GetNodeAt(X, Y);
  if (Node <> nil) and
    ((Node <> DropTarget) or (Node = FLastDropTarget)) then
  begin
    FLastDropTarget := nil;
    TDragObject(Source).HideDragImage;
    if CanDrop then TreeView_SelectDropTarget(Handle, Node)
    else if GetItemState(Node, nsDropHilited) then TreeView_SelectDropTarget(Handle, nil);
    TDragObject(Source).ShowDragImage;
  end;
end;

{procedure TCustomDBTreeView.GetImageIndex(Node: HTreeItem);
begin
  if Assigned(FOnGetImageIndex) then FOnGetImageIndex(Self, Node);
end;}

{procedure TCustomDBTreeView.GetSelectedIndex(Node: HTreeItem);
begin
  if Assigned(FOnGetSelectedIndex) then FOnGetSelectedIndex(Self, Node);
end;}

function TCustomDBTreeView.CanChange(Node: HTreeItem): Boolean;
begin
  Result := True;
  if Assigned(FOnChanging) then FOnChanging(Self, Node, Result);
end;

procedure TCustomDBTreeView.Change(Node: HTreeItem);
begin
  if Assigned(FOnChange) then FOnChange(Self, Node);
end;

procedure TCustomDBTreeView.Expand(Node: HTreeItem);
begin
  if Assigned(FOnExpanded) then FOnExpanded(Self, Node);
end;

function TCustomDBTreeView.CanExpand(Node: HTreeItem): Boolean;
begin
  Result := True;
  if Assigned(FOnExpanding) then FOnExpanding(Self, Node, Result);
  ExpandItem(Node, False);
end;

procedure TCustomDBTreeView.Collapse(Node: HTreeItem);
begin
  if Assigned(FOnCollapsed) then FOnCollapsed(Self, Node);
end;

function TCustomDBTreeView.CanCollapse(Node: HTreeItem): Boolean;
begin
  Result := True;
  if Assigned(FOnCollapsing) then FOnCollapsing(Self, Node, Result);
  TreeView_Expand(Handle, node, TVE_COLLAPSE or TVE_COLLAPSERESET);
end;

function TCustomDBTreeView.CanEdit(Node: HTreeItem): Boolean;
begin
  Result := True;
  if Assigned(FOnEditing) then FOnEditing(Self, Node, Result);
end;

procedure TCustomDBTreeView.Edit(const Item: TTVItem);
var
  S: string;
  Node: HTreeItem;
begin
  with Item do
    if pszText <> nil then
    begin
      S := pszText;
      Node := Item.hItem;
      if Assigned(FOnEdited) then FOnEdited(Self, Node, S);
      //if Node <> nil then Node.Text := S;
    end;
end;

procedure TCustomDBTreeView.SetImageList(Value: HImageList; Flags: Integer);
begin
  if HandleAllocated then TreeView_SetImageList(Handle, Value, Flags);
end;

procedure TCustomDBTreeView.ImageListChange(Sender: TObject);
var
  ImageHandle: HImageList;
begin
  if HandleAllocated then
  begin
    ImageHandle := TImageList(Sender).Handle;
    if Sender = Images then
      SetImageList(ImageHandle, TVSIL_NORMAL)
    else if Sender = StateImages then
      SetImageList(ImageHandle, TVSIL_STATE);
  end;
end;

procedure TCustomDBTreeView.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if Operation = opRemove then
  begin
    if AComponent = Images then Images := nil;
    if AComponent = StateImages then StateImages := nil;
  end;
end;

procedure TCustomDBTreeView.SetImages(Value: TImageList);
begin
  if Images <> nil then
    Images.UnRegisterChanges(FImageChangeLink);
  FImages := Value;
  if Images <> nil then
  begin
    Images.RegisterChanges(FImageChangeLink);
    SetImageList(Images.Handle, TVSIL_NORMAL)
  end
  else SetImageList(0, TVSIL_NORMAL);
end;

procedure TCustomDBTreeView.SetStateImages(Value: TImageList);
begin
  if StateImages <> nil then
    StateImages.UnRegisterChanges(FStateChangeLink);
  FStateImages := Value;
  if StateImages <> nil then
  begin
    StateImages.RegisterChanges(FStateChangeLink);
    SetImageList(StateImages.Handle, TVSIL_STATE)
  end
  else SetImageList(0, TVSIL_STATE);
end;

procedure TCustomDBTreeView.WMRButtonDown(var Message: TWMRButtonDown);
var
  MousePos: TPoint;
begin
  FRClickNode := nil;
  try
    if not RightClickSelect then
    begin
      inherited;
      if FRClickNode <> nil then
      begin
        GetCursorPos(MousePos);
        with PointToSmallPoint(ScreenToClient(MousePos)) do
          Perform(WM_RBUTTONUP, 0, MakeLong(X, Y));
      end;
    end
    else DefaultHandler(Message);
  finally
    FRClickNode := nil;
  end;
end;

procedure TCustomDBTreeView.WMRButtonUp(var Message: TWMRButtonUp);

  procedure DoMouseDown(var Message: TWMMouse; Button: TMouseButton;
    Shift: TShiftState);
  begin
    if not (csNoStdEvents in ControlStyle) then
      with Message do
        MouseDown(Button, KeysToShiftState(Keys) + Shift, XPos, YPos);
  end;

begin
  if RightClickSelect then DoMouseDown(Message, mbRight, []);
  inherited;
end;

procedure TCustomDBTreeView.WMLButtonDown(var Message: TWMLButtonDown);
var
  Node: HTreeItem;
  MousePos: TPoint;
begin
  FDragged := False;
  FDragNode := nil;
  try
    inherited;
    if DragMode = dmAutomatic then
    begin
      SetFocus;
      if not FDragged then
      begin
        GetCursorPos(MousePos);
        with PointToSmallPoint(ScreenToClient(MousePos)) do
          Perform(WM_LBUTTONUP, 0, MakeLong(X, Y));
      end
      else begin
        Node := GetNodeAt(Message.XPos, Message.YPos);
        if Node <> nil then
        begin
           FocusItem(Node, True);
           TreeView_SelectItem(Handle, Node);
          BeginDrag(False);
        end;
      end;
    end;
  finally
    FDragNode := nil;
  end;
end;

procedure TCustomDBTreeView.WMNotify(var Message: TWMNotify);
var
  Node: HTreeItem;
  MaxTextLen: Integer;
  Pt: TPoint;
begin
  with Message do
    if NMHdr^.code = TTN_NEEDTEXTW then
    begin
      // Work around NT COMCTL32 problem with tool tips >= 80 characters
      GetCursorPos(Pt);
      Pt := ScreenToClient(Pt);
      Node := GetNodeAt(Pt.X, Pt.Y);
      if (Node = nil) then Exit;
      FWideText := '?';
      MaxTextLen := SizeOf(PToolTipTextW(NMHdr)^.szText) div SizeOf(WideChar);
      if Length(FWideText) >= MaxTextLen then
        SetLength(FWideText, MaxTextLen - 1);
      PToolTipTextW(NMHdr)^.lpszText := PWideChar(FWideText);
      FillChar(PToolTipTextW(NMHdr)^.szText, MaxTextLen, 0);
      Move(Pointer(FWideText)^, PToolTipTextW(NMHdr)^.szText, Length(FWideText));
      PToolTipTextW(NMHdr)^.hInst := 0;
      SetWindowPos(NMHdr^.hwndFrom, HWND_TOP, 0, 0, 0, 0, SWP_NOACTIVATE or
        SWP_NOSIZE or SWP_NOMOVE);
      Result := 1;
    end
    else inherited;
end;

function TCustomDBTreeView.GetChildren(Item: HTreeItem): Boolean;
var
  TVItem: TTVItem;
begin
  TVItem.mask := TVIF_CHILDREN;
  TVItem.hItem := Item;
  if TreeView_GetItem(Handle, TVItem) then Result := TVItem.cChildren > 0
  else Result := False;
end;

function TCustomDBTreeView.HasAsParent(Item, Value: HTreeItem): Boolean;
begin
  if Value <> Nil then
  begin
    if TreeView_GetParent(Handle, Item) = Nil then Result := False
    else if TreeView_GetParent(Handle, Item) = Value then Result := True
    else Result := HasAsParent(TreeView_GetParent(Handle, Item), Value);
  end
  else Result := True;
end;

procedure TCustomDBTreeView.DeleteItem(Item: HTreeItem);
begin
  if Assigned(FOnDeletion) then
    FOnDeletion(Self, Item);
  if FLastDropTarget = Item then
    FLastDropTarget := nil;
  if Item <> nil then TreeView_DeleteItem(Handle, Item);
end;

(*function TCustomDBTreeView.GetNextItem(Item: HTreeItem): HTreeItem;
var
  NodeID, ParentID: HTreeItem;
begin
  NodeID := TreeView_GetChild(Handle, Item);
  if NodeID = nil then
    NodeID := TreeView_GetNextSibling(Handle, Item);
  ParentID := Item;
  while (NodeID = nil) and (ParentID <> nil) do
  begin
    ParentID := TreeView_GetParent(Handle, ParentID);
    NodeID := TreeView_GetNextSibling(Handle, ParentID);
  end;
  Result := NodeID;
end;*)

procedure TCustomDBTreeView.ExpandItem(Item: HTreeItem; Recurse: Boolean);
Var
  i: Integer;
begin
  FDetailTable.Filtered:= False;
  if ( Item = nil ) then begin
    if ( FDataLink.DataSet.Filter <> '' ) then
      FDetailTable.Filter:= FDataLink.DataSet.Filter+' and '+FDetailFieldName+'='+FStartMasterValue
    else
      FDetailTable.Filter:= FDetailFieldName+'='+FStartMasterValue;
  end
  else begin
    if ( FDataLink.DataSet.Filter <> '' ) then
      FDetailTable.Filter:= FDataLink.DataSet.Filter+' and '+FDetailFieldName+'='+IntToStr(GetItem(Item).lParam)
    else
      FDetailTable.Filter:= FDetailFieldName+'='+IntToStr(GetItem(Item).lParam);
  end;
  FDetailTable.Filtered:= True;
  FDetailTable.First;
  for i:= 0 to FDetailTable.RecordCount - 1 do begin
    AddItem(Item, TVI_LAST, FDetailTable.FieldByName(FMasterFieldName).AsInteger);
    FDetailTable.Next;
  end;
end;

procedure TCustomDBTreeView.CollapseItem(Item: HTreeItem; Recurse: Boolean);
var
  NodeID, ParentID: HTreeItem;
begin
  NodeID := TreeView_GetChild(Handle, Item);
  if NodeID = nil then Exit;
  ParentID:= NodeID;
  while (NodeID <> nil) do
  begin
    NodeID := TreeView_GetNextSibling(Handle, ParentID);
    DeleteItem(ParentID);
    ParentID:= NodeID;
  end;
end;

function TCustomDBTreeView.GetItemState(Item: HTreeItem; NodeState: TNodeState): Boolean;
var
  TVItem: TTVItem;
begin
  Result := False;
  with TVItem do
  begin
    mask := TVIF_STATE;
    hItem := Item;
    if TreeView_GetItem(Handle, TVItem) then
      case NodeState of
        nsCut: Result := (state and TVIS_CUT) <> 0;
        nsFocused: Result := (state and TVIS_FOCUSED) <> 0;
        nsSelected: Result := (state and TVIS_SELECTED) <> 0;
        nsExpanded: Result := (state and TVIS_EXPANDED) <> 0;
        nsDropHilited: Result := (state and TVIS_DROPHILITED) <> 0;
      end;
  end;
end;

procedure TCustomDBTreeView.FocusItem(Item: HTreeItem; Value: Boolean);
var
  TVItem: TTVItem;
  Template: Integer;
begin
  if Value then Template := -1
  else Template := 0;
  with TVItem do
  begin
    mask := TVIF_STATE;
    hItem := Item;
    stateMask := TVIS_FOCUSED;
    state := stateMask and Template;
  end;
  TreeView_SetItem(Handle, TVItem);
end;

function TCustomDBTreeView.AddItem(Parent, Target: HTreeItem; Code: Longint): HTreeItem;
var
  InsertStruct: TTVInsertStruct;
  TVItem: TTVItem;
begin
  with InsertStruct do
  begin
    hParent := Parent;
    hInsertAfter := Target;
  end;
  with TVItem do
  begin
    mask := TVIF_TEXT or TVIF_PARAM or TVIF_IMAGE or TVIF_SELECTEDIMAGE or TVIF_CHILDREN;
    lParam := Code;
    pszText := LPSTR_TEXTCALLBACK;
    iImage := I_IMAGECALLBACK;
    iSelectedImage := I_IMAGECALLBACK;
    cChildren := I_CHILDRENCALLBACK;
  end;
  InsertStruct.item := TVItem;
  Result:= TreeView_InsertItem(Handle, InsertStruct);
end;

(*function TCustomDBTreeView.GetChildrenCount(Item: HTreeItem): Integer;
var
  Node: HTreeItem;
begin
  Result := 0;
  Node := TreeView_GetChild(Handle, Item);
  while Node <> nil do
  begin
    Inc(Result);
    Node := TreeView_GetNextSibling(Handle, Node);
  end;
end;*)

function TCustomDBTreeView.GetItem(Item: HTreeItem): TTVItem;
begin
  with Result do begin
    mask:= TVIF_HANDLE;
    hItem:= Item;
    TreeView_GetItem(Handle, Result);
  end;
end;

function TCustomDBTreeView.FindItemData(Code: Longint): HTreeItem;
Var
  TVItem: TTVItem;
begin
  with TVItem do begin
    mask:= TVIF_PARAM;
    lParam:= Code;
    hItem:= nil;
    TreeView_GetItem(Handle, TVitem);
    Result:= TVItem.hItem;
  end;
end;

procedure TCustomDBTreeView.SetHotTrack(Value: Boolean);
begin
  if FHotTrack <> Value then
  begin
    FHotTrack := Value;
    RecreateWnd;
  end;
end;

procedure TCustomDBTreeView.SetCheckboxes(Value: Boolean);
begin
  if FCheckboxes <> Value then
  begin
    FCheckboxes := Value;
    RecreateWnd;
    LinkActive(FDataLink.Active);
  end;
end;

procedure TCustomDBTreeView.SetToolTips(Value: Boolean);
begin
  if FToolTips <> Value then
  begin
    FToolTips := Value;
    RecreateWnd;
  end;
end;

procedure TCustomDBTreeView.RecordChanged(Field: TField);
begin
  if not HandleAllocated or FUpdateLock then Exit;
  if ValidDataSet and ( FDataLink.DataSet.State = dsBrowse ) then
    Invalidate;

end;

function TCustomDBTreeView.AcquireFocus: Boolean;
begin
  Result := True;
  if FAcquireFocus and CanFocus and not (csDesigning in ComponentState) then
  begin
    SetFocus;
    Result := Focused;
  end;
end;

procedure TCustomDBTreeView.LinkActive(Value: Boolean);
begin
  if not Value then HideEditor;
  if ( ValidDataSet ) then begin
    if ( MasterFieldName = '' ) or ( FDataLink.DataSet.FindField(MasterFieldName) = nil ) then begin
      MessageDlg(StrMasterFieldNameEmpty, mtWarning, [mbOk], 0 );
      Exit;
    end;
    if ( DetailFieldName = '' ) or ( FDataLink.DataSet.FindField(MasterFieldName) = nil ) then begin
      MessageDlg(StrDetailFieldNameEmpty, mtWarning, [mbOk], 0 );
      Exit;
    end;
    if ( ItemFieldName = '' ) then begin
      MessageDlg(StrItemFieldNameEmpty, mtWarning, [mbOk], 0 );
      Exit;
    end;
    {if ( IconFieldName = '' ) then begin
      MessageDlg(StrIconFieldNameEmpty, mtWarning, [mbOk], 0 );
      Exit;
    end;}
    if ( FDataLink.DataSet.FindField(MasterFieldName).DataType <> FDataLink.DataSet.FindField(DetailFieldName).DataType ) then begin
      MessageDlg(StrMasterDetailFieldNameError, mtWarning, [mbOk], 0 );
      Exit;
    end;
    if ( FDataLink.DataSet.FindField(ItemFieldName).DataType in
        [ftBytes,ftVarBytes,ftBlob,ftMemo,ftGraphic,ftFmtMemo,ftParadoxOle,ftDBaseOle,ftTypedBinary]) then begin
      MessageDlg(StrItemFieldNameError, mtWarning, [mbOk], 0 );
      Exit;
    end;
    if ( IconFieldName <> '' ) and Not ( FDataLink.DataSet.FindField(IconFieldName).DataType in [ftSmallInt, ftInteger, ftWord] ) then begin
      MessageDlg(StrIconFieldNameError, mtWarning, [mbOk], 0 );
      Exit;
    end;
    if ( ColorFieldName <> '' ) and ( FDataLink.DataSet.FindField(ColorFieldName) <> Nil ) and
       Not ( FDataLink.DataSet.FindField(ColorFieldName).DataType in [ftSmallInt, ftInteger, ftWord] ) then begin
      MessageDlg(StrColorFieldNameError, mtWarning, [mbOk], 0 );
      Exit;
    end;
  end;
  //if ( csDesigning in ComponentState ) then Exit;
  if (ValidDataSet) and ( ( FDataLink.DataSet is TTable ) )  then begin
    FDetailTable.Active:= False;
    FDetailTable.DataBaseName:= (FDataLink.DataSet as TTable).DataBaseName;
    FDetailTable.TableName:= (FDataLink.DataSet as TTable).TableName;
    FDetailTable.IndexFieldNames:= (FDataLink.DataSet as TTable).IndexFieldNames;
    FDetailTable.Active:= True;
    ExpandItem(nil, False);
  end
  else
    TreeView_DeleteAllItems(Handle);
end;

procedure TCustomDBTreeView.DataChanged;
begin
  if not HandleAllocated or FUpdateLock then Exit;
  {if ( FUpdateLock = 0 ) then
    UpdateRowCount;}
  UpdateActive;
  //if ( FUpdateLock = 0 ) then begin
  if ValidDataSet and ( FDataLink.DataSet.State = dsBrowse ) then begin
    ValidateRect(Handle, nil);
    Invalidate;
  end;

  //end;
end;

(*procedure TCustomDBTreeView.UpdateRowCount;

begin
  if Not ValidDataSet then Exit;
  //if Count <= 1 then Count:= 1;
  with FDataLink do
    if not Active or (RecordCount = 0) or not HandleAllocated then
      //Count := 0
    else
    begin
      //Count := FDataLink.DataSet.RecordCount;
      FDataLink.BufferCount := FDataLink.DataSet.RecordCount;
      //Count; //VisibleCount;
      UpdateActive;
    end;
end;*)

function TCustomDBTreeView.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TCustomDBTreeView.SetDataSource(Value: TDataSource);
begin
  if Value = FDatalink.Datasource then Exit;
  TreeView_DeleteAllItems(Handle);
  FOldFrom:= -1;
  FDataLink.DataSource := Value;
  if Value <> nil then Value.FreeNotification(Self);
  //LinkActive(FDataLink.Active);
end;

procedure TCustomDBTreeView.ScrollData(Distance: Integer);
begin
  UpdateActive;
  //ListView_EnsureVisible(Handle, Selected, TRUE);
end;

function TCustomDBTreeView.ValidDataSet: Boolean;
begin
  Result:= False;
  if ( DataLink <> nil ) And Assigned(DataLink.DataSet) And DataLink.DataSet.Active then
    Result:= True;
end;

function TCustomDBTreeView.SelectedField: TField;
begin
  Result:= nil;
  if ( ValidDataSet ) then
    Result:= FDataLink.DataSet.FindField(FItemFieldName);
end;

procedure TCustomDBTreeView.ShowEditor;
begin
  TreeView_EditLabel(Handle, Selected);
end;

procedure TCustomDBTreeView.HideEditor;
begin
  SendMessage(Handle, WM_CANCELMODE, 0, 0);
end;

procedure TCustomDBTreeView.EditingChanged;
begin
end;

procedure TCustomDBTreeView.UpdateData;
begin
end;

procedure TCustomDBTreeView.UpdateActive;
//var
  //NewRow: Integer;
begin
  {if FDatalink.Active and HandleAllocated and not (csLoading in ComponentState) then
  begin
    NewRow := FDatalink.ActiveRecord;
    if Selected <> NewRow then
    begin
      HideEditor;
      SetSelection(NewRow+TopItem);
    end;
  end;}
end;

procedure TCustomDBTreeView.KeyDown(var Key: Word; Shift: TShiftState);
var
  KeyDownEvent: TKeyEvent;

  function DeletePrompt: Boolean;
  var
    Msg: string;
  begin
  Msg := SDeleteRecordQuestion;
    Result := {not (dgConfirmDelete in Options) or}
      (MessageDlg(Msg, mtConfirmation, mbOKCancel, 0) <> idCancel);
  end;

const
  RowMovementKeys = [VK_UP, VK_PRIOR, VK_DOWN, VK_NEXT, VK_HOME, VK_END];

begin
  KeyDownEvent := OnKeyDown;
  if Assigned(KeyDownEvent) then KeyDownEvent(Self, Key, Shift);
  if not ValidDataSet then Exit;
  with FDatalink.DataSet do
    if ssCtrl in Shift then
    begin
      case Key of
        VK_DELETE:
          if Not GetChildren(Selected) then begin
            if (not ReadOnly) and not IsEmpty
              and CanModify and DeletePrompt then
              Delete;
          end;
      end
    end
    else
      case Key of
        {VK_INSERT:
          if CanModify and (not ReadOnly) then
          begin
            ClearSelection;
            NewItem;
          end;}
        VK_ESCAPE:
          begin
            FDatalink.Reset;
            HideEditor;
          end;
        VK_F2: ShowEditor;
        VK_F5: begin
            TreeView_DeleteItem(Handle, nil);
            ExpandItem(nil, False);
          end;
      end;
  inherited;
end;

procedure TCustomDBTreeView.SetMasterFieldName(Value: String);
Var AField: TField;
begin
  AField:= nil;
  if ValidDataSet and
    not (csLoading in ComponentState) and (Length(Value) > 0) then
      AField := DataLink.DataSet.FindField(Value); { no exceptions }
  if ( AField <> nil ) and Not ( AField.DataType in [ftSmallInt, ftInteger, ftWord] ) then begin
    MessageDlg(StrMasterFieldNameError, mtWarning, [mbOk], 0 );
    Value:= '';
    AField:= nil;
  end;
  FMasterFieldName:= Value;
end;

procedure TCustomDBTreeView.SetColorFieldName(Value: String);
Var AField: TField;
begin
  AField:= nil;
  if ValidDataSet and
    not (csLoading in ComponentState) and (Length(Value) > 0) then
      AField := DataLink.DataSet.FindField(Value); { no exceptions }
  if ( AField <> nil ) and Not ( AField.DataType in [ftSmallInt, ftInteger, ftWord] ) then begin
    MessageDlg(StrColorFieldNameError, mtWarning, [mbOk], 0 );
    Value:= '';
  end;
  FColorFieldName:= Value;
end;

procedure TCustomDBTreeView.SetDetailFieldName(Value: String);
Var AField: TField;
begin
  AField:= nil;
  if ValidDataSet and
    not (csLoading in ComponentState) and (Length(Value) > 0) then
      AField := DataLink.DataSet.FindField(Value); { no exceptions }
  if ( AField <> nil ) and ( AField.DataType in
        [ftBytes,ftVarBytes,ftBlob,ftMemo,ftGraphic,ftFmtMemo,ftParadoxOle,ftDBaseOle,ftTypedBinary]) then begin
    MessageDlg(StrDetailFieldNameError, mtWarning, [mbOk], 0 );
    Value:= '';
  end;
  FDetailFieldName:= Value;
end;

procedure TCustomDBTreeView.SetItemFieldName(Value: String);
Var AField: TField;
begin
  AField:= nil;
  if ValidDataSet and
    not (csLoading in ComponentState) and (Length(Value) > 0) then
      AField := DataLink.DataSet.FindField(Value); { no exceptions }
  if ( AField <> nil ) and ( AField.DataType in
        [ftBytes,ftVarBytes,ftBlob,ftMemo,ftGraphic,ftFmtMemo,ftParadoxOle,ftDBaseOle,ftTypedBinary]) then begin
    MessageDlg(StrItemFieldNameError, mtWarning, [mbOk], 0 );
    Value:= '';
  end;
  FItemFieldName:= Value;
end;

procedure TCustomDBTreeView.SetIconFieldName(Value: String);
Var AField: TField;
begin
  AField:= nil;
  if ValidDataSet and
    not (csLoading in ComponentState) and (Length(Value) > 0) then
      AField := DataLink.DataSet.FindField(Value); { no exceptions }
  if ( AField <> nil ) and Not ( AField.DataType in [ftSmallInt, ftInteger, ftWord] ) then begin
    MessageDlg(StrIconFieldNameError, mtWarning, [mbOk], 0 );
    Value:= '';
    AField:= nil;
  end;
  FIconFieldName:= Value;
end;

end.
